﻿Imports Microsoft.Win32
Imports System.Text
Imports System.Runtime.InteropServices

Public Class HybridConfig
    ' iniまたはレジストリへ設定を保存・読出しするためのハイブリット設定クラス
    Private m_rKey As RegistryKey
    Private m_strIniFile As String
    Private m_strSection As String

    ' デフォルトコンストラクタ
    Public Sub HybridConfig()
        m_rKey = Nothing
        m_strIniFile = Nothing
        m_strSection = Nothing
    End Sub

    ' レジストリ用コンストラクタ。第1引数はRegistryクラスでopenしたキー
    Public Sub HybridConfig(ByVal rKey As RegistryKey)
        m_strIniFile = Nothing
        m_strSection = Nothing
        OpenRegKey(rKey)
    End Sub

    ' iniファイル用コンストラクタ。iniファイル名とセクション名を渡す
    Public Sub HybridConfig(ByVal strIniFile As String, ByVal strSection As String)
        m_rKey = Nothing
        OpenIniFile(strIniFile, strSection)
    End Sub

    ' レジストリ用open。第1引数はRegistryクラスでopenしたキー
    Public Function OpenRegKey(ByVal rKey As RegistryKey) As Boolean
        If rKey Is Nothing Then
            ' 引数不正
            Return False
        Else
            ' Registryクラスでopen処理はされているので、それを代入するだけ
            m_rKey = rKey
            Return True
        End If
    End Function

    ' iniファイル用open。iniファイル名とセクション名を渡す
    Public Function OpenIniFile(ByVal strIniFile As String, ByVal strSection As String) As Boolean
        If strIniFile Is Nothing OrElse strSection Is Nothing Then
            ' 引数不正
            Return False
        Else
            ' この時点では読出し／書き込みなどは行われないので、ファイル名とセクション名を代入するだけ
            m_strIniFile = strIniFile
            m_strSection = strSection
            Return True
        End If
    End Function

    ' 共通クローズ
    Public Sub Close()
        If Not m_rKey Is Nothing Then
            m_rKey.Close()
            m_rKey = Nothing
        End If
        m_strIniFile = Nothing
        m_strSection = Nothing
    End Sub

    ' 共通：Integer型の値の読出し
    Public Function GetIntValue(ByVal strValueName As String, ByVal intDefValue As Integer) As Integer
        If Not m_rKey Is Nothing Then
            Return CInt(m_rKey.GetValue(strValueName, intDefValue))
        ElseIf (Not m_strIniFile Is Nothing) AndAlso (Not m_strSection Is Nothing) Then
            Return win32api.GetPrivateProfileInt(m_strSection, strValueName, intDefValue, m_strIniFile)
        Else
            ' 未初期化で呼び出された:デフォルト値を返す
            Return intDefValue
        End If
    End Function

    ' 共通：Boolean型の値の読出し
    Public Function GetBolValue(ByVal strValueName As String, ByVal bolDefValue As Boolean) As Boolean
        Dim intDefValue As Integer
        Dim intRetValue As Integer

        If bolDefValue Then
            intDefValue = 1
        Else
            intDefValue = 0
        End If

        intRetValue = GetIntValue(strValueName, intDefValue)
        If intRetValue = 0 Then
            Return False
        Else
            Return True
        End If
    End Function

    ' 共通：String型の値の読出し
    Public Function GetStrValue(ByVal strValueName As String, ByVal strDefValue As String) As String
        If Not m_rKey Is Nothing Then
            Return CStr(m_rKey.GetValue(strValueName, strDefValue))
        ElseIf (Not m_strIniFile Is Nothing) AndAlso (Not m_strSection Is Nothing) Then
            Dim stbReturnValue As StringBuilder = New StringBuilder(1024)
            win32api.GetPrivateProfileString(m_strSection, strValueName, strDefValue, stbReturnValue, CUInt(stbReturnValue.Capacity), m_strIniFile)

            Return stbReturnValue.ToString()
        Else
            ' 未初期化で呼び出された:デフォルト値を返す
            Return strDefValue
        End If
    End Function

    ' 共通：Integer型の値の書き込み
    Public Function SetIntValue(ByVal strValueName As String, ByVal intValue As Integer) As Integer
        If Not m_rKey Is Nothing Then
            m_rKey.SetValue(strValueName, intValue)
            Return 1
        ElseIf (Not m_strIniFile Is Nothing) AndAlso (Not m_strSection Is Nothing) Then
            ' initファイルへ書き込みを行うAPIにint型のものは無いため、文字列化して書き込む
            If 0 = win32api.WritePrivateProfileString(m_strSection, strValueName, intValue.ToString(), m_strIniFile) Then
                Throw New Exception("Cannot write to " + m_strIniFile + ".")
            End If
            Return 1
        Else
            ' 未初期化で呼び出された:エラー扱い
            Return 0
        End If
    End Function

    ' 共通：Boolean型の値の書き込み
    Public Function SetBolValue(ByVal strValueName As String, ByVal bolValue As Boolean) As Integer
        Dim intValue As Integer

        If bolValue Then
            intValue = 1
        Else
            intValue = 0
        End If

        Return SetIntValue(strValueName, intValue)
    End Function

    ' 共通：String型の値の書き込み
    Public Function SetStrValue(ByVal strValueName As String, ByVal strValue As String) As Integer
        If (Not m_rKey Is Nothing) Then
            m_rKey.SetValue(strValueName, strValue)
            Return 1
        ElseIf (Not m_strIniFile Is Nothing) AndAlso (Not m_strSection Is Nothing) Then
            If 0 = win32api.WritePrivateProfileString(m_strSection, strValueName, strValue, m_strIniFile) Then
                Throw New Exception("Cannot write to " + m_strIniFile + ".")
            End If
            Return 1
        Else
            ' 未初期化で呼び出された:エラー扱い
            Return 0
        End If
    End Function
End Class

Public Class win32api
    <DllImport("KERNEL32.DLL")> _
    Public Shared Function WritePrivateProfileString( _
        ByVal lpAppName As String, _
        ByVal lpKeyName As String, _
        ByVal lpString As String, _
        ByVal lpFileName As String) As Integer
    End Function

    <DllImport("KERNEL32.DLL", CharSet:=CharSet.Auto)> _
    Public Shared Function GetPrivateProfileString( _
        ByVal lpAppName As String, _
        ByVal lpKeyName As String, ByVal lpDefault As String, _
        ByVal lpReturnedString As System.Text.StringBuilder, ByVal nSize As UInteger, _
        ByVal lpFileName As String) As Integer
    End Function

    <DllImport("KERNEL32.DLL", CharSet:=CharSet.Auto)> _
    Public Shared Function GetPrivateProfileInt( _
        ByVal lpAppName As String, _
        ByVal lpKeyName As String, ByVal nDefault As Integer, _
        ByVal lpFileName As String) As Integer
    End Function
End Class
